/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.fingerprint.authentication.nocipher;

import android.content.Context;
import android.hardware.fingerprint.FingerprintManager;
import android.os.CancellationSignal;

public class FingerprintAuthentication {

    private FingerprintManager mFingerprintManager;
    private CancellationSignal mCancellationSignal;


    public FingerprintAuthentication(Context context) {
        mFingerprintManager = (FingerprintManager) context.getSystemService(Context.FINGERPRINT_SERVICE);
        reset();
    }

    public boolean startAuthentication(final FingerprintManager.AuthenticationCallback callback) {

        mCancellationSignal = new CancellationSignal();

        // Callback which accepts the result of fingerprint authentication
        FingerprintManager.AuthenticationCallback hook = new FingerprintManager.AuthenticationCallback() {
                @Override
                public void onAuthenticationError(int errorCode, CharSequence errString) {
                    if (callback != null) callback.onAuthenticationError(errorCode, errString);
                    reset();
                }

                @Override
                public void onAuthenticationHelp(int helpCode, CharSequence helpString) {
                    if (callback != null) callback.onAuthenticationHelp(helpCode, helpString);
                }

                @Override
                public void onAuthenticationSucceeded(FingerprintManager.AuthenticationResult result) {
                    if (callback != null) callback.onAuthenticationSucceeded(result);
                    reset();
                }

                @Override
                public void onAuthenticationFailed() {
                    if (callback != null) callback.onAuthenticationFailed();
                }
            };

        // Perform fingerprint authentication
        // By giving null as the first argument CryptoObject, the authentication is not linked with key
        mFingerprintManager.authenticate(null, mCancellationSignal, 0, hook, null);

        return true;
    }

    public boolean isAuthenticating() {
        return mCancellationSignal != null && !mCancellationSignal.isCanceled();
    }

    public void cancel() {
        if (mCancellationSignal != null) {
            if (!mCancellationSignal.isCanceled())
                mCancellationSignal.cancel();
        }
    }

    private void reset() {
        mCancellationSignal = null;
    }

    public boolean isFingerprintAuthAvailable() {
        return (mFingerprintManager.hasEnrolledFingerprints()) ? true : false;
    }

    public boolean isFingerprintHardwareDetected() {
        return mFingerprintManager.isHardwareDetected();
    }


}
